unit RecPageProd;

{
  TRecordPageProducer - a page producer that iterates through a data set
  and generates the attached snippet for each record.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 17 August 1999.
  Updated 12 December, 2002.
}

interface

{$I Defines.inc}

uses
  SysUtils, Classes,
{$IFDEF D6UP}
  HTTPProd,
{$ELSE}
  HTTPApp,
{$ENDIF}
  Db;

type
  TRecordPageProducer = class(TPageProducer)
  private
    FDataSet: TDataSet;
    FNoRecsDoc: TStrings;
    FNoRecsFile: TFileName;
    procedure SetDataSet(DataSet: TDataSet);
    procedure SetNoRecsDoc(Value: TStrings);
    procedure SetNoRecsFile(const Value: TFileName);
  protected
    procedure DoTagEvent(Tag: TTag; const TagString: string;
      TagParams: TStrings; var ReplaceText: string); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ContentFromStream(Stream: TStream): string; override;
  published
    property DataSet: TDataSet read FDataSet write SetDataSet;
    property NoRecsDoc: TStrings read FNoRecsDoc write SetNoRecsDoc;
    property NoRecsFile: TFileName read FNoRecsFile write SetNoRecsFile;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Internet', [TRecordPageProducer]);
end;

{ TRecordPageProducer ---------------------------------------------------------}

{ Initialisation }
constructor TRecordPageProducer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FNoRecsDoc := TStringList.Create;
end;

{ Release resources }
destructor TRecordPageProducer.Destroy;
begin
  FNoRecsDoc.Free;
  inherited Destroy;
end;

{ Clear our reference to the data set if it is deleted }
procedure TRecordPageProducer.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and Assigned(DataSet) and
      (DataSet = AComponent) then
    DataSet := nil;
end;

{ Reference the data set }
procedure TRecordPageProducer.SetDataSet(DataSet: TDataSet);
begin
  if FDataSet <> DataSet then
  begin
    if Assigned(DataSet) then
      DataSet.FreeNotification(Self);
    FDataSet := DataSet;
  end;
end;

{ Set the no records template as embedded text }
procedure TRecordPageProducer.SetNoRecsDoc(Value: TStrings);
begin
  FNoRecsDoc.Assign(Value);
  FNoRecsFile := '';
end;

{ Set the no records template as a file reference }
procedure TRecordPageProducer.SetNoRecsFile(const Value: TFileName);
begin
  if CompareText(FNoRecsFile, Value) <> 0 then
  begin
    FNoRecsDoc.Clear;
    FNoRecsFile := Value;
  end;
end;

{ Iterate through the records in the dataset }
function TRecordPageProducer.ContentFromStream(Stream: TStream): string;
var
  stmNoRecs: TStream;
begin
  Result := '';
  if Assigned(FDataSet) then
    if FDataSet.Active then
      if FDataSet.RecordCount > 0 then
        { Cycle through all the records }
        with FDataSet do
        begin
          First;
          while not EOF do
          begin
            Stream.Position := 0;
            Result := Result + inherited ContentFromStream(Stream);
            Next;
          end;
          Exit;
        end;

  { No data found }
  if FNoRecsFile <> '' then
    stmNoRecs := TFileStream.Create(FNoRecsFile, fmOpenRead + fmShareDenyWrite)
  else
    stmNoRecs := TStringStream.Create(FNoRecsDoc.Text);
  if Assigned(stmNoRecs) then
  try
    Result := inherited ContentFromStream(stmNoRecs);
  finally
    stmNoRecs.Free;
  end;
end;

{ Replace field references automatically }
procedure TRecordPageProducer.DoTagEvent(Tag: TTag; const TagString: string;
  TagParams: TStrings; var ReplaceText: string);
begin
  try
    ReplaceText := FDataSet.FieldByName(TagString).DisplayText;
  except
    inherited DoTagEvent(Tag, TagString, TagParams, ReplaceText);
  end;
end;

end.
